/*
 * Author: Heinz Mauelshagen, Germany
 *
 * March-May,October 1997
 * May 1998
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *    10/11/1997 - added display of missing physical volumes
 *    10/12/1997 - added trying deactivation of inconsistent volume groups
 *    11/10/1997 - used lvm_tab
 *    05/09/1998 - implemented -x option
 *    05/16/1998 - added lvmtab checking
 *
 */

#include <lvm_user.h>

#ifdef DEBUG
int opt_d = 0;
#endif

int main ( int argc, char **argv) {
   int backup = FALSE;
   int c = 0;
   int p = 0;
   int ret = 0;
   int set_extendable = FALSE;
   int v = 0;
   char *vg_name = NULL;
#ifdef DEBUG
   char *options = "A:a:dh?vx:";
#else
   char *options = "A:a:h?vx:";
#endif
   char **vg_name_ptr = NULL;
   int opt_A = 1;
   int opt_a = 0;
   int opt_a_action = 0;
   int opt_v = 0;
   int opt_x = 0;
   int opt_x_action = 0;
   char *cmd = NULL;
   pv_t **pv_offl = NULL;
   pv_t **pv_incons = NULL;
   vg_t *vg = NULL;

   cmd = basename ( argv[0]);

   SUSER_CHECK;
   LVMTAB_CHECK;

   while ( ( c = getopt ( argc, argv, options)) != EOF) {
      switch ( c) {
         case 'A':
            if ( opt_A > 1) {
               fprintf ( stderr, "%s -- A option yet given\n\n", cmd);
               return 1;
            }
            opt_A++;
            if ( strcmp ( optarg, "y") == 0);
            else if ( strcmp ( optarg, "n") == 0) opt_A = 0;
            else {
               fprintf ( stderr, "%s -- invalid option argument %s\n\n",
                                 cmd, optarg);
               return 1;
            }
            break;

         case 'a':
            if ( opt_a > 0) {
               fprintf ( stderr, "%s -- a option yet given\n\n", cmd);
               return 1;
            }
            opt_a++;
            if ( strcmp ( optarg, "y") == 0) {
               opt_a_action = VG_CREATE;
               break;
            } else if ( strcmp ( optarg, "n") == 0) {
               opt_a_action = ~VG_CREATE;
               break;
            }
            fprintf ( stderr, "%s -- a option argument \"%s\" invalid\n\n",
                      cmd, optarg);
            return 1;
            break;

#ifdef DEBUG
         case 'd':
            if ( opt_d > 0) {
               fprintf ( stderr, "%s -- d option yet given\n\n", cmd);
               return 1;
            }
            opt_d++;
            break;
#endif

         case 'h':
         case '?':
            printf ( "\n%s\n\n%s -- Volume group change\n\n"
                     "Synopsis:\n"
                     "---------\n\n"
                     "%s\n"
                     "\t[-A y/n]\n"
                     "\t[-a y/n [VolumeGroupName...]]\n"
#ifdef DEBUG
                     "\t[-d]\n"
#endif
                     "\t[-h/-?]\n"
                     "\t[-v]\n"
                     "\t[-x y/n [VolumeGroupName...]]\n\n",
                     lvm_version, cmd, cmd);
            return 0;
            break;

         case 'x':
            if ( opt_x > 0) {
               fprintf ( stderr, "%s -- x option yet given\n\n", cmd);
               return 1;
            }
            opt_x++;
            if ( strcmp ( optarg, "y") == 0) {
               opt_x_action = VG_EXTENDABLE;
               break;
            } else if ( strcmp ( optarg, "n") == 0) {
               opt_x_action = ~VG_EXTENDABLE;
               break;
            }
            fprintf ( stderr, "%s -- a option argument \"%s\" invalid\n\n",
                      cmd, optarg);
            return 1;
            break;

         case 'v':
            if ( opt_v > 0) {
               fprintf ( stderr, "%s -- v option yet given\n\n", cmd);
               return 1;
            }
            opt_v++;
            break;

         default:
            fprintf ( stderr, "%s -- invalid command line option \"%c\"\n\n",
                      cmd, c);
            return 1;
      }
   }
  
   if ( opt_a == 0 && opt_x == 0) {
      fprintf ( stderr, "%s -- you have to give the -a or -x option\n\n", cmd);
      return 1;
   }
   
   if ( opt_a == 1 && opt_A > 1) {
      fprintf ( stderr, "%s -- -A option not neccessary with "
                        "-a option\n\n", cmd);
      return 1;
   }
   
   if ( optind < argc) {
      if ( opt_v > 0) printf ( "%s -- using volume group(s) on command line\n",
                               cmd);
      vg_name_ptr = argv + optind;
      argc = argc - optind;
   } else {
      if ( opt_v > 0) printf ( "%s -- finding all volume group(s)\n", cmd);
      vg_name_ptr = lvm_tab_vg_check_exist_all_vg ();
      argc = 0;
      if ( vg_name_ptr != NULL)
         for ( v = 0; vg_name_ptr[v] != NULL; v++) argc++;
   }
   argv = vg_name_ptr;
   optind = 0;

   if ( optind == argc) {
      printf ( "%s -- no volume groups found\n\n", cmd);
      return 0;
   }

   LVM_LOCK ( 0);
   LVM_CHECK_IOP;

   /* work on all given/found volume groups */
   for ( ; optind < argc; optind++) {
      vg_name = argv[optind];

      if ( opt_v > 0) printf ( "%s -- checking volume group name %s\n",
                               cmd, vg_name);
      if ( vg_check_name ( vg_name) < 0) {
         fprintf ( stderr, "%s -- invalid volume group name %s\n",
                   cmd, vg_name);
         continue;
      }
   
      if ( opt_v > 0) printf ( "%s -- checking existence of volume group %s\n",
                               cmd, vg_name);
      if ( ( ret = lvm_tab_vg_check_exist ( vg_name, NULL)) < 0) {
         if ( ret == -LVM_EPV_READ_PV_EXPORTED)
            fprintf ( stderr, "%s -- can't access exported volume group %s\n",
                              cmd, vg_name);
         else if ( ret == -LVM_EVG_CHECK_EXIST_PV_COUNT)
            fprintf ( stderr, "%s -- ERROR: not all physical volumes of "
                              "volume group %s online\n", cmd, vg_name);
         else if ( ret == -LVM_EPV_READ_ALL_PV_OF_VG_NP)
            fprintf ( stderr, "%s -- volume group %s doesn't exist\n",
                      cmd, vg_name);
         else if ( ret == -LVM_EVG_READ_LVM_STRUCT_VERSION)
            fprintf ( stderr, "%s -- volume group %s has physical volumes "
                              "with invalid version\n",
                      cmd, vg_name);
         else
            fprintf ( stderr, "%s -- ERROR %d checking existence of %s\n",
                      cmd, ret, vg_name);

         if ( vg_check_active ( vg_name) == TRUE) {
            printf ( "%s -- trying to deactivate inconsistent "
                     "volume group %s\n", cmd, vg_name);
            if ( ( ret = vg_remove ( vg_name)) < 0) {
               fprintf ( stderr, "%s -- ERROR %d trying to deactivate %s\n",
                                   cmd, ret, vg_name);
            } else {
               printf ( "%s -- volume group %s deactivated\n", cmd, vg_name);
            }
         }
         continue;
      }

      if ( opt_v > 0) printf ( "%s -- reading volume group data for %s "
                               "from lvmtab\n", cmd, vg_name);
      if ( ( ret = lvm_tab_vg_read_with_pv_and_lv ( vg_name, &vg)) < 0) {
         fprintf ( stderr, "%s -- ERROR %d: reading lvmtab data of %s\n",
                   cmd, ret, vg_name);
         continue;
      }
   
      if ( opt_v > 0) printf ( "%s -- checking volume group consistency "
                               " of %s\n", cmd, vg_name);
      if ( ( ret = vg_check_consistency_with_pv_and_lv ( vg)) < 0) {
         fprintf ( stderr, "%s -- ERROR %d: %s is inconsistent\n",
                   cmd, ret, vg_name);
         continue;
         if ( vg_check_active ( vg_name) == TRUE) {
            printf ( "%s -- trying to deactivate inconsistent "
                     "volume group %s\n", cmd, vg_name);
            if ( ( ret = vg_remove ( vg_name)) < 0) {
               fprintf ( stderr, "%s -- ERROR %d trying to deactivate %s\n",
                                   cmd, ret, vg_name);
            } else {
               printf ( "%s -- volume group %s deactivated\n", cmd, vg_name);
            }
         }
      }

      lvm_dont_interrupt ( 0);
   
      if ( opt_a > 0) {
         if ( opt_a_action == VG_CREATE) {
            if ( opt_v > 0) printf ( "%s -- checking if all physical volumes "
                                     "of %s are available\n",
                                     cmd, vg_name);
            if ( ( ret = vg_check_online_all_pv ( vg, &pv_offl,
                                                  &pv_incons)) < 0 &&
                 ret != -LVM_EPV_READ_MD_DEVICE) {
               for ( p = 0; pv_offl != NULL && pv_offl[p] != NULL; p++) {
                  fprintf ( stderr, "%s -- ERROR: physical volume %s of "
                                    "volume group %s is offline\n",
                                    cmd, pv_offl[p]->pv_name, vg_name);
               }
               for ( p = 0; pv_incons != NULL && pv_incons[p] != NULL; p++) {
                  fprintf ( stderr, "%s -- ERROR: physical volume %s of "
                                    "volume group %s is inconsistent\n",
                                    cmd, pv_incons[p]->pv_name, vg_name);
               }
               printf ( "%s -- run vgscan\n", cmd);
               continue;
            }
            if ( pv_offl != NULL) free ( pv_offl);
            if ( pv_incons != NULL) free ( pv_incons);
      
            if ( opt_v > 0) printf ( "%s -- creating VGDA for %s in kernel\n",
                                     cmd, vg_name);
            if ( ( ret = vg_create ( vg_name, vg)) == 0)
               printf ( "%s -- %s successfully activated\n", cmd, vg_name);
            else {
   #ifdef DEBUG
               debug ( "%s -- vg_create returned: %d\n", cmd, ret);
   #endif
               if ( ret == -LVM_EVG_CREATE_REMOVE_OPEN) {
                  fprintf ( stderr, "%s -- can't open logical volume manager "
                                    "special to activate %s\n", cmd, vg_name);
               } else if ( ret == -EPERM) {
                  fprintf ( stderr, "%s -- volume group %s is already active\n",
                                    cmd, vg_name);
               } else if ( ret == -ENOMEM) {
                  fprintf ( stderr, "%s -- memory error activating %s\n",
                                    cmd, vg_name);
               } else {
                  fprintf ( stderr, "%s -- ERROR %d activating %s\n",
                  cmd, ret, vg_name);
               }
               continue;
            }
         } else {
            if ( opt_v > 0) printf ( "%s -- removing VGDA for %s from kernel\n",
                                     cmd, vg_name);
            if ( ( ret = vg_remove ( vg_name)) == 0)
               printf ( "%s -- %s successfully deactivated\n", cmd, vg_name);
            else {
   #ifdef DEBUG
               debug ( "%s -- vg_remove returned: %d\n", cmd, ret);
   #endif
               if ( ret == -LVM_EVG_CREATE_REMOVE_OPEN) {
                  fprintf ( stderr, "%s -- can't open logical volume manager "
                                    "special to deactivate %s\n?\n", cmd, vg_name);
               } else if ( ret == -EPERM) {
                   fprintf ( stderr, "%s -- can't deactivate volume group %s "
                                     "with open logical volume(s)\n",
                                     cmd, vg_name);
               } else if ( ret == -ENXIO) {
                   fprintf ( stderr, "%s -- volume group %s isn't active\n",
                             cmd, vg_name);
               } else {
                   fprintf ( stderr, "%s -- ERROR %d DEactivating %s\n",
                             cmd, ret, vg_name);
               }
               continue;
            }
         }
      }

      backup = FALSE;
      if ( opt_x > 0) {
         if ( opt_x_action == VG_EXTENDABLE) {
            if ( vg->vg_status & VG_EXTENDABLE) {
               printf ( "%s -- volume group %s ist yet extendable\n",
                        cmd, vg->vg_name);
            } else {
               vg->vg_status |= VG_EXTENDABLE;
               set_extendable = TRUE;
               backup = TRUE;
            }
         } else {
            if ( ! ( vg->vg_status & VG_EXTENDABLE)) {
               printf ( "%s -- volume group %s ist allready not extendable\n",
                        cmd, vg_name);
            } else {
               vg->vg_status &= ~VG_EXTENDABLE;
               set_extendable = FALSE;
               backup = TRUE;
            }
         }
         if ( backup == TRUE) {
            /* store vg on disk(s) */
            if ( opt_v > 0) printf ( "%s -- storing volume group data of "
                                     "%s on disk(s)\n", cmd, vg_name);
            if ( ( ret = vg_write_with_pv_and_lv ( vg)) < 0) {
               fprintf ( stderr, "%s -- ERROR %d storing volume group data "
                                 "of %s on disk(s)\n\n", cmd, ret, vg_name);
               return 1;
            }
      
            if ( opt_v > 0) printf ( "%s -- changing VGDA in kernel\n", cmd);
            if ( set_extendable == TRUE) {
               ret = vg_set_extendable ( vg_name);
            } else {
               ret = vg_clear_extendable ( vg_name);
            }
            if ( ret < 0) {
               fprintf ( stderr, "%s -- ERROR %d ", cmd, ret);
               if ( set_extendable == TRUE) {
                   fprintf ( stderr, "en");
               } else {
                   fprintf ( stderr, "dis");
              }
              fprintf ( stderr, "abling extension of %s\n\n", vg_name);
              return 1;
            }

            if ( opt_v > 0) printf ( "%s -- changing lvmtab\n", cmd);
            if ( ( ret = vg_cfgbackup ( vg_name,
                                        LVMTAB_DIR, cmd, opt_v, vg)) == 0 &&
                 opt_A > 0) {
               printf ( "%s -- doing automatic backup of %s\n", cmd, vg_name);
               ret = vg_cfgbackup ( vg_name, VG_BACKUP_DIR, cmd, opt_v, vg);
            }
            if ( ret == 0) {
               printf ( "%s -- volume group %s successfully changed\n\n",
                        cmd, vg_name);
            } else {
               fprintf ( stderr, "%s -- ERROR %d storing volume group backup "
                                 "of %s on disk(s)\n\n", cmd, ret, vg_name);
               return 1;
            }
         }
      }
      lvm_interrupt ();
   }

   printf ( "\n");

   LVM_UNLOCK ( 0);

   if ( ret == 0) return 0;
   else           return 1;
}
